/*!
 * @file game_world.h
 *  container for all game worlds (singleplayers, multiplayers..)
 */

#ifndef _GAME_WORLD_H
#define _GAME_WORLD_H


#include "story_entity.h"
#include "game_world_data.h"
#include "playable.h"
#include "script_manager.h"

namespace OrxShell { class Shell; };
class WorldEntity;
class GameRules;


/** How many frames time values to keep
 * The higher the value the smoother the result is...
 * Don't make it 0 or less :)
 */
#define TICK_SMOOTH_VALUE 10

//! The game world
/**
 *  this class initializes everything that should be displayed inside of the current level.
 *  it is the main driving factor during gameplay.
 */
class GameWorld : public StoryEntity
{
  ObjectListDeclaration(GameWorld);

public:
  GameWorld ();
  virtual ~GameWorld ();

  virtual void loadParams(const TiXmlElement* root);

  /* functions from story-entity */
  virtual ErrorMessage init();
  virtual ErrorMessage loadData();
  virtual ErrorMessage unloadData();

  virtual bool start();
  virtual bool stop();
  virtual bool pause();
  virtual bool resume();
  virtual void run();

  void setPlaymode(Playable::Playmode playmode);
  void setPlaymode(const std::string& playmode);
  /**  this returns the current game time @returns elapsed game time     */
  inline double getGameTime() { return this->gameTime; }
  /** sets the game speed @param speed speed of the Game */
  inline void setSpeed(float speed) { this->speed = speed; };
  /**  returns the track path of this world @returns the track path */


  void setSoundtrack(const std::string& soundTrack);
  void togglePNodeVisibility();
  void toggleBVVisibility(int level);
  inline void toggleMPVisibility() { this->showMPV = !this->showMPV; }

  inline void setSky(WorldEntity* sky) { this->dataTank->sky = sky; }


protected:
  /* world - running functions */
  virtual void synchronize();
  virtual void handleInput();
  virtual void tick(ObjectManager::EntityList worldEntity, float dt);
  virtual void tick();
  virtual void update();
  virtual void checkGameRules();
  virtual void collisionDetection();
  virtual void collisionReaction();

  void applyCameraSettings();
  void drawEntityList(const ObjectManager::EntityList& drawList ) const;
  virtual void renderPassReflection();
  virtual void renderPassRefraction();
  virtual void renderPassAll();


  virtual void display();


private:
  void displayLoadScreen();
  void releaseLoadScreen();


protected:
  GameWorldData*      dataTank;                     //!< reference to the GameWorld Data Tank
  TiXmlElement*       dataXML;                      //!< The XML-Element this World has been loaded with.

  bool                showPNodes;                   //!< if the PNodes should be visible.
  bool                showBV;                       //!< if the Bounding Volumes should be visible.
  int                 showBVLevel;                  //!< the depth level of the displayed bounding volumes
  bool                showMPV;                      //!< true if the mounting points should be drawn for debug purposes

  /* world timing */
  double              lastFrame;                    //!< last time of frame (in MiliSeconds)
  Uint32              cycle;                        //!< The cycle we are in (starts with 0 and rises with every frame)
  float               dtS;                          //!< The time needed for caluculations in seconds
  float               speed;                        //!< how fast the game flows
  double              gameTime;                     //!< this is where the game time is saved
  double              frameTimes[TICK_SMOOTH_VALUE];//!< The time used for the last TICK_SMOOTH_VALUE's frames.

  GameRules*          gameRules;                    //!< Pointer to the data structure containig the game rules

private:
  /* external modules interfaces */
  ScriptManager       scriptManager;
  OrxShell::Shell*    shell;
};

#endif /* _GAME_WORLD_H */
