/*!
 * @file shader.h
 * @brief Definition of the Shader rendering class
*/

#ifndef _SHADER_H
#define _SHADER_H

#include "base_object.h"
#include "shader_data.h"

// FORWARD DECLARATION



//! The Shader is a Class-wrapper around the OpenGL Shader Language (GLSL).
class Shader : public BaseObject
{
  ObjectListDeclaration(Shader);
public:
  class Uniform
  {
  public:
    Uniform(const Shader& shader, const std::string& location) { if (Shader::isSupported()) this->uniform = glGetUniformLocation(shader.getProgram(), location.c_str()) ; };
  Uniform(GLhandleARB shaderProgram, const std::string& location) { if (Shader::isSupported()) this->uniform = glGetUniformLocation(shaderProgram, location.c_str()) ; };

  void set(float v0) const { if (Shader::isSupported()) glUniform1f(this->uniform, v0); }
  void set(float v0, float v1) const { if (Shader::isSupported()) glUniform2f(this->uniform, v0, v1); }
  void set(float v0, float v1, float v2) const { if (Shader::isSupported()) glUniform3f(this->uniform, v0, v1, v2); }
  void set(float v0, float v1, float v2, float v3) const { if (Shader::isSupported()) glUniform4f(this->uniform, v0, v1, v2, v3); }

  void set(int v0) const { if (Shader::isSupported()) glUniform1i(this->uniform, v0); }
  void set(int v0, int v1) const { if (Shader::isSupported()) glUniform2i(this->uniform, v0, v1); }
  void set(int v0, int v1, int v2) const { if (Shader::isSupported()) glUniform3i(this->uniform, v0, v1, v2); }
  void set(int v0, int v1, int v2, int v3) const {if (Shader::isSupported())  glUniform4i(this->uniform, v0, v1, v2, v3); }

    void setV(unsigned int count, float* vv) const;
    void setV(unsigned int count, int* vv) const;

  private:
    GLint uniform;
  };


public:
  Shader();
  Shader(const Shader& shader);
  Shader(const std::string& vertexShaderFile, const std::string& fragmentShaderFile = "");

  Shader& operator=(const Shader& shader) { this->data = shader.data; return *this; };
  const ShaderData::Pointer& dataPointer() const { return data; };
  void acquireData(const ShaderData::Pointer& pointer) { this->data = pointer; };

  bool load(const std::string& vertexShaderFile, const std::string& fragmentShaderFile = "")
  { return this->data->load(vertexShaderFile, fragmentShaderFile); }


  Shader::Uniform getUniform(const std::string& location) { return Shader::Uniform(*this, location); }

  GLhandleARB getProgram() const { return this->data->getProgram(); }
  GLhandleARB getVertexS() const { return this->data->getVertexS(); }
  GLhandleARB getFragmentS() const { return this->data->getFragmentS(); }

  void activateShader() const;
  static void deactivateShader();

  void debug() const { this->data->debug(); };


  static bool checkShaderAbility();
  static bool isSupported() { return checkShaderAbility(); };

inline static bool isShaderActive() { return (Shader::storedShader != NULL) ? true : false; };
  inline static const Shader* getActiveShader() { return Shader::storedShader; };

  inline static void suspendShader() { const Shader* currShader = storedShader; if (storedShader!= NULL) { Shader::deactivateShader(); Shader::storedShader = currShader;} };
  inline static void restoreShader() { if (storedShader != NULL) storedShader->activateShader(); storedShader = NULL; };

private:
  ShaderData::Pointer          data;

  static const Shader*         storedShader;
};

#endif /* _SHADER_H */
