/*!
 * @file static_model.h
 * @brief Contains the Model Class that handles Static 3D-Models rendered with glList's
 */

#ifndef _STATIC_MODEL_DATA_H
#define _STATIC_MODEL_DATA_H

#include "model.h"

#include "material.h"
#include <vector>
#include "count_pointer.h"


// definition of different modes for setting up Faces
#define VERTEX 0       //!< If Faces are created WITH Vertex-Coordinate
#define NORMAL 1       //!< If Faces are created WITH Normals (otherwise autocalculate)
#define TEXCOORD 2     //!< If Faces are created WITH TextureCoordinate

//! an enumerator for VERTEX_FORMAT
typedef enum VERTEX_FORMAT {
  VERTEX_ONLY = VERTEX,
  VERTEX_NORMAL = NORMAL,
  VERTEX_TEXCOORD = TEXCOORD,
  VERTEX_TEXCOORD_NORMAL = NORMAL | TEXCOORD
};



/////////////
/// MODEL ///
/////////////
//! Class that handles static 3D-Models.
/**
 * it can also read them in and display them.
 * All the objects are rendered with glLists
 */
class StaticModelData : public BaseObject
{
  ObjectListDeclaration(StaticModelData);

private:
  ////////////////////
  /// SUB-ELEMENTS ///
  ////////////////////
  //! This is the placeholder of one Vertex beloning to a Face.
  class FaceElement
  {
  public:
    FaceElement();

    int                 vertexNumber;         //!< The number of the Vertex out of the Array* vertices, this vertex points to.
    int                 normalNumber;         //!< The number of the Normal out of the Array* normals, this vertex points to.
    int                 texCoordNumber;       //!< The number of the textureCoordinate out of the Array* vTexture, this vertex points to.
  };

  //! This is the placeholder of a Face belonging to a Group of Faces.
  class Face
  {
  public:
    Face();

    std::vector<FaceElement>  _elements; //!< Elements of the Face.
    Material*                 _material;        //!< The Material to use.
  };

  //! Group to handle multiple Models per obj-file.
  class Group
  {
  public:
    Group();
    ~Group();

    //! Compares the name with the groups name.
    bool operator==(const std::string& name) const { return this->name == name; };
    void cleanup();

    std::string  name;           //!< the Name of the Group. this is an identifier, that can be accessed via the draw (std::string name) function.
    GLubyte*     indices;        //!< The indices of the Groups. Needed for vertex-arrays
    GLuint       listNumber;     //!< The number of the GL-List this Group gets.
    int          faceMode;       //!< The Mode the Face is in: initially -1, 0 for FaceList opened, 1 for Material,  3 for triangle, 4 for Quad, 5+ for Poly @todo ENUM...

    std::vector<Face> _faces;    //!< Faces.
  };

  public:
  typedef CountPointer<StaticModelData> Pointer;

public:
  StaticModelData(const std::string& modelName = "");
  virtual ~StaticModelData();

  void draw() const;
  void draw(unsigned int groupNumber) const;
  void draw(const std::string& groupName) const;

  void rebuild();

  Material* addMaterial(const Material& material);
  Material* addMaterial(const std::string& materialName);

  bool addGroup(const std::string& groupString);

  void setScaleFactor(float scaleFactor) { this->scaleFactor = scaleFactor; };

  bool addVertex(const std::string& vertexString);
  bool addVertex(float x, float y, float z);

  bool addFace(const std::string& faceString);
  bool addFace(int faceElemCount, VERTEX_FORMAT type, va_list args);

  bool addVertexNormal(const std::string& normalString);
  bool addVertexNormal(float x, float y, float z);

  bool addVertexTexture(const std::string& vTextureString);
  bool addVertexTexture(float u, float v);

  bool setMaterial(const std::string& mtlString);
  bool setMaterial(Material* mtl);

  void finalize();


  const std::vector<GLfloat>& getVertices() const { return this->vertices; };
  const std::vector<GLfloat>& getNormals() const { return this->normals; };
  const std::vector<GLfloat>& getTexCoords() const { return this->vTexture; };
  const std::vector<sTriangleExt>& getTriangles() const { return this->triangles; };
  ///! HACK SOLUTION sTriangleExt should be const in the modelInfo.
  sTriangleExt* getTrianglesExt() { return &this->triangles[0]; };

  float getScaleFactor() const  { return scaleFactor; }

protected:
  Material* findMaterialByName(const std::string& materialName);

private:
  bool buildVertexNormals();

  bool importToDisplayList();
  bool buildTriangleList();

  bool addGLElement(const StaticModelData::FaceElement& elem);

  bool cleanup();

private:
  bool                       finalized;       //!< Sets the Object to be finalized.

  float                      scaleFactor;     //!< The Factor with which the Model should be scaled. @todo maybe one wants to scale the Model after Initialisation

  unsigned int               faceCount;       //!< A modelwide Counter for the faces

  std::vector<GLfloat>       vertices;        //!< The Array that handles the Vertices.
  std::vector<GLfloat>       normals;         //!< The Array that handles the Normals.
  std::vector<GLfloat>       vTexture;        //!< The Array that handles the VertexTextureCoordinates.

  std::vector<sTriangleExt>  triangles;       //!< The Triangles if built.


  std::vector<Group>         _modelGroups;

  std::list<Material>        materialList;    //!< A list for all the Materials in this Model
};

#endif /* _STATIC_MODEL_DATA_H */
