/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: Christoph Renner
   co-programmer: ...
*/

#include "player_stats.h"

#include "src/lib/util/loading/factory.h"

#include "player.h"
#include "playable.h"
#include "state.h"
#include "shared_network_data.h"

#include "converter.h"

#include "parser/preferences/preferences.h"

#include "debug.h"
#include "shell_command.h"



ObjectListDefinition(PlayerStats);
CREATE_FACTORY(PlayerStats);

/**
 * constructor
 */
PlayerStats::PlayerStats( int userId )
{
  init();

  this->assignedUserId = userId;
}

/**
 * constructor
 */
PlayerStats::PlayerStats( const TiXmlElement* root )
{
  init();
}

void PlayerStats::init( )
{
  this->registerObject(this, PlayerStats::_objectList);

  this->assignedUserId = 0;
  this->teamId = TEAM_NOTEAM;
  this->preferedTeamId = TEAM_NOTEAM;
  this->score = 0;
  this->playableClassId = 0;
  this->modelFileName = "";
  this->nickName = "Player";
  this->oldNickName = "Player";

  userId_handle = registerVarId( new SynchronizeableInt( &assignedUserId, &assignedUserId, "userId", PERMISSION_MASTER_SERVER ) );
  teamId_handle = registerVarId( new SynchronizeableInt( &teamId, &teamId, "teamId", PERMISSION_MASTER_SERVER ) );
  preferedTeamId_handle = registerVarId( new SynchronizeableInt( &preferedTeamId, &preferedTeamId, "preferedUserId", PERMISSION_MASTER_SERVER ) );
  score_handle = registerVarId( new SynchronizeableInt( &score, &score, "score", PERMISSION_MASTER_SERVER ) );
  playableClassId_handle = registerVarId( new SynchronizeableInt( &playableClassId, &playableClassId, "playableClassId", PERMISSION_MASTER_SERVER) );
  playableUniqueId_handle = registerVarId( new SynchronizeableInt( &playableUniqueId, &playableUniqueId, "playableUniqueId", PERMISSION_MASTER_SERVER ) );
  modelFileName_handle = registerVarId( new SynchronizeableString( &modelFileName, &modelFileName, "modelFileName", PERMISSION_MASTER_SERVER ) );
  nickName_handler = registerVarId( new SynchronizeableString( &nickName, &nickName, "nickName", PERMISSION_MASTER_SERVER ) );

  MessageManager::getInstance()->registerMessageHandler( MSGID_CHANGENICKNAME, changeNickHandler, NULL );

  PRINTF(5)("PlayerStats created\n");
}


/**
 * standard deconstructor
 */
PlayerStats::~PlayerStats()
{}


/**
* override this function to be notified on change
* of your registred variables.
* @param id id's which have changed
 */
void PlayerStats::varChangeHandler( std::list< int > & id )
{
  if ( std::find( id.begin(), id.end(), playableUniqueId_handle ) != id.end() )
  {
    this->setPlayableUniqueId( this->playableUniqueId );

    PRINTF(4)("uniqueID changed %d %d %d\n", assignedUserId, SharedNetworkData::getInstance()->getHostID(), getUniqueID());
  }

  if ( std::find( id.begin(), id.end(), nickName_handler ) != id.end() )
  {
    PRINTF(0)("user %s is now known as %s\n", oldNickName.c_str(), nickName.c_str());
    this->oldNickName = nickName;
  }

  if ( std::find( id.begin(), id.end(), preferedTeamId_handle) != id.end() )
  {
    PRINTF(0)("user %s has changed team to %i\n", nickName.c_str(), this->teamId);
  }
}



/**
 * get stats for user userId
 * @param userId user's id
 * @return stats assigned to user
 */
PlayerStats * PlayerStats::getStats( int userId )
{
  for (ObjectList<PlayerStats>::const_iterator it = PlayerStats::objectList().begin();
       it != PlayerStats::objectList().end();
       ++it)
  {
    if ( (*it)->getAssignedUserId() == userId )
    {
      return (*it);
    }
  }

  return NULL;
}

/**
 * set playable class id and set playable
 */
void PlayerStats::setPlayableUniqueId( int uniqueId )
{
  this->playable = NULL;
  for (ObjectList<Playable>::const_iterator it = Playable::objectList().begin();
       it != Playable::objectList().end();
       ++it)
  {
    if ( (*it)->getUniqueID() == uniqueId )
    {
      this->playable = (*it);
      //TODO when OM_PLAYERS is ticked add line:
      //this->playable->toList( OM_PLAYERS );
      break;
    }
  }

  if ( this->playable && this->assignedUserId == SharedNetworkData::getInstance()->getHostID() )
  {
    State::getPlayer()->setPlayable( this->playable );
    // also set the team id
  }

  this->playableUniqueId = uniqueId;
}

/**
 * get playable associated to player
 * @return playable associated to player
 */
Playable * PlayerStats::getPlayable()
{
  if ( playable )
    return playable;

  assert( playableUniqueId > 0 );

  setPlayableUniqueId( playableUniqueId );

  assert( playable );

  return playable;
}

/**
 * client sends server a message to change nick and server changes nick directly
 * @param nick new nickname
 */
void PlayerStats::setNickName( std::string nick )
{
  if ( SharedNetworkData::getInstance()->isMasterServer())
  {
    this->nickName = nick;
    PRINTF(0)("user %s is now known as %s\n", oldNickName.c_str(), nickName.c_str());
    oldNickName = nickName;
    return;
  }
  else
  {
    byte * data = new byte[nick.length()+INTSIZE];

    assert( Converter::stringToByteArray( nick, data, nick.length()+INTSIZE) == nick.length()+INTSIZE );

    MessageManager::getInstance()->sendMessage( MSGID_CHANGENICKNAME, data, nick.length()+INTSIZE, RT_SERVER, NET_MASTER_SERVER, MP_HIGHBANDWIDTH );
    return;
  }
}

/**
 * handler for the nick name
 * @param messageType type of the message
 * @param data  data of the message
 * @param dataLength length of the data
 * @param someData some additional data
 * @param senderId userId of the sender
 * @param destinationId userId of the rec
 * @return true if handled correctly
 */
bool PlayerStats::changeNickHandler( MessageType messageType, byte * data, int dataLength, void * someData, int senderId, int destinationId  )
{
  std::string newNick;
  int res = Converter::byteArrayToString( data, newNick, dataLength );

  if ( res != dataLength )
  {
    PRINTF(2)("invalid message size from user %d\n", senderId);
    newNick = "invalid";
  }

  if ( PlayerStats::getStats( senderId) )
    PlayerStats::getStats( senderId )->setNickName( newNick );

  return true;
}

/**
 * sets the nick name from the shell
 * @param newNick new prefered nick name
 */
void PlayerStats::shellNick( const std::string& newNick )
{
  if ( getStats( SharedNetworkData::getInstance()->getHostID() ) )
    getStats( SharedNetworkData::getInstance()->getHostID() )->setNickName( newNick );

  Preferences::getInstance()->setString( "multiplayer", "nickname", newNick );
}



/**
 * removes and delets all player stats
 */
void PlayerStats::deleteAllPlayerStats( )
{

  while(!PlayerStats::objectList().empty())
    delete PlayerStats::objectList().front();
}



/**
 * @return the score list of this player stat
 */
ScoreList PlayerStats::getScoreList( )
{
  ScoreList result;

  for (ObjectList<PlayerStats>::const_iterator it = PlayerStats::objectList().begin();
       it != PlayerStats::objectList().end();
       ++it)
  {
    PlayerStats& stats = *(*it);

    TeamScoreList::iterator it = result[stats.getTeamId()].begin();

    while (  it != result[stats.getTeamId()].end() && stats.score > it->score )
    {
      it++;
    }

    PlayerScore score;
    score.name = stats.getNickName();
    score.score = stats.getScore();

    result[stats.getTeamId()].insert(it, score);
  }

  return result;
}
