/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: Patrick Boenzli
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_COLLISION_REACTION

#include "collision_tube.h"

#include "world_entity.h"

#include "collision.h"
#include "collision_event.h"
#include "collision_reaction.h"

#include "cr_object_damage.h"
#include "cr_physics_ground_walk.h"
#include "cr_physics_full_walk.h"

#include "debug.h"


namespace CoRe
{

  ObjectListDefinition(CollisionTube);

  CollisionTube* CollisionTube::singletonRef = NULL;


  /**
   * standard constructor
   * @todo this constructor is not jet implemented - do it
   */
  CollisionTube::CollisionTube ()
  {
    this->registerObject(this, CollisionTube::_objectList);
  }


  /**
   * standard deconstructor
   */
  CollisionTube::~CollisionTube ()
  {
    this->reset();
  }



  /**
   * resets the collision tube after each collision detection cycle
   */
  void CollisionTube::reset()
  {
    this->_collisionList.clear();
  }


  /**
   * gets the collision object useable for collision events registarations
   * @param entityA collision partner 1
   * @param entityB collision partner 2
   * @return the collision object to be used
   */
  Collision* CollisionTube::registerCollision(WorldEntity* entityA, WorldEntity* entityB)
  {
    assert( entityA != NULL && entityB != NULL);

    Collision* collision = NULL;
    if( !this->_collisionList.empty())
      collision = this->_collisionList.back();

    // check if there is already a collision defined between these objects or this is the first collision at all
    if( collision == NULL || !collision->same(*entityA, *entityB))
    {
      collision = CREngine::getInstance()->popCollisionObject();
      collision->collide( entityA, entityB);
      this->_collisionList.push_back(collision);
    }

    return collision;
  }


  /**
   * registers a new CollisionEvent
   * @param entityA one collision object
   * @param entityB the other collision objectName
   * @param bvA bounding volume of object A
   * @param bvB bounding volume of object B
   *
   * this function doesn't check if the entities in question actualy are registered for any collisions
   */
  void CollisionTube::registerCollisionEvent(WorldEntity* entityA, WorldEntity* entityB, BoundingVolume* bvA, BoundingVolume* bvB)
  {
    // first get a useable collision object
    Collision* collision = this->registerCollision(entityA, entityB);

    // now register the new collision event
    CollisionEvent* collisionEvent = CREngine::getInstance()->popCollisionEventObject();
    collisionEvent->collide( CREngine::CR_COLLISION_TYPE_OBB, entityA, entityB, bvA, bvB);
    collision->registerCollisionEvent( collisionEvent);
  }


  /**
   * registers a new CollisionEvent, only used by ground to object collision (eg. bsp model)
   * @param type type of collision as stated in cr_def.h
   * @param entity the WorldEntity colliding with the ground
   * @param groundEntity the WorldEntity representing the ground
   * @param normal the normal vector for the ground (up) - not always specified
   * @param position the position of the collision relative to the object center
   * @param bInWall true if the entity is in the ground material
   */
  void CollisionTube::registerCollisionEvent(CREngine::CollisionType type, WorldEntity* entity, WorldEntity* groundEntity,
      const Vector& normal, const Vector& position, bool bInWall)
  {
    // first get a useable collision object
    Collision* collision = this->registerCollision(entity, groundEntity);

    // now register the new collision event
    CollisionEvent* collisionEvent = CREngine::getInstance()->popCollisionEventObject();
    collisionEvent->collide( type, entity, groundEntity, normal, position, bInWall);
    collision->registerCollisionEvent( collisionEvent);
  }



}// namespace end

