/*!
 * @file collision.h
 *  Definition of a collision relation of two WorldEntities
 *
 *  Is shared between two WorldEntity's CollisionHandles if both are subscribed to this event. In this case only one
 *  of the two CollisionHandles will calculate the CollisionReaction and the bDispatched flag will be set afterwards
 *  to signal that it's already cared about and should be ignored.
 *
 *  The collisions itself are saved in this container (CollisionEvent). Since there can be multiple collision events
 *  for one collision. Imagine: two objects are intersecting (this throws a Collision): many collision boxes will fire
 *  each of this boxes will "create" a CollisionEvent.
 */

#ifndef _COLLISION_H
#define _COLLISION_H

#include "vector.h"
#include <vector>



class WorldEntity;
class BoundingVolume;

namespace CoRe
{

  class CollisionEvent;

  //! A class representing a simple collision
  class Collision
  {

    typedef std::vector<CollisionEvent*>                  CollisionVector;  //!< the vector containing all collision events
    typedef std::vector<CollisionEvent*>::iterator        Iterator;         //!< iterator definition
    typedef std::vector<CollisionEvent*>::const_iterator  ConstIterator;   //!< constant iterator definition


  public:

    Collision();
    virtual ~Collision();

    /** collides two WorldEntities @param entityA world entity A, @param entityB world entity B, @param bvA volume A @param bvB volumeB */
    inline void collide(WorldEntity* entityA, WorldEntity* entityB) { this->_entityA = entityA; this->_entityB = entityB; }


    /* list stuff */
  public:

    /** registers a @param event CollisionEvent to take place */
    inline void registerCollisionEvent(CollisionEvent* event) { this->_collisionEvents.push_back(event); }

    void reset();


    /** @returns iterator pointing to the beginning of the collision event vector */
    inline Iterator       begin()        { return this->_collisionEvents.begin(); }
    /** @returns const iterator pointing to the beginning of the collision event vector */
    inline ConstIterator  begin() const  { return this->_collisionEvents.begin(); }
    /** @returns iterator pointing to the end of the collision event vector */
    inline Iterator       end()          { return this->_collisionEvents.end(); }
    /** @returns const iterator pointing to the end of the collision event vector */
    inline ConstIterator  end() const    { return this->_collisionEvents.end(); }


    /* misc interface functions */
  public:
    /** @return Collision WorldEntity A */
    inline WorldEntity* getEntityA() const { return this->_entityA; }
    /** @return Collision WorldEntity B */
    inline WorldEntity* getEntityB() const { return this->_entityB; }
    inline bool same(const WorldEntity& entityA, WorldEntity& entityB) const {
      return ((this->_entityA == &entityA && this->_entityB == &entityB) || (this->_entityA == &entityB && this->_entityB == &entityA)); }

    /** @return true if Entity A collides */
    inline bool isEntityACollide() const { return this->_entityACollide; }
    /** sets the flag if it reacts @param flag true if it should react on entityA*/
    inline void setEntityACollide(bool flag) { this->_entityACollide = flag; }
    /** @return true if Entity B collides */
    inline bool isEntityBCollide() const { return this->_entityBCollide; }
    /** sets the flag if it reacts @param flag true if it should react on entityB*/
    inline void setEntityBCollide(bool flag) { this->_entityBCollide = flag; }


  private:
    WorldEntity*                 _entityA;                       //!< the collision body A
    WorldEntity*                 _entityB;                       //!< the collision body B
    bool                         _entityACollide;                //!< true if entity A is subscribed for collision reaction
    bool                         _entityBCollide;                //!< true if entity B is subscribed for collision reaction

    CollisionVector              _collisionEvents;               //!< the collision event list

  };
}

#endif /* _COLLISION_H */
